<?php

/**
 * i-doit
 *
 * Auth: Class for i-doit authorization rules.
 *
 * @package     i-doit
 * @subpackage  auth
 * @author      Selcuk Kekec <skekec@i-doit.com>
 * @version     1.2.0
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       i-doit 1.2.0
 */
class isys_auth_document extends isys_auth implements isys_auth_interface
{
    /**
     * Container for singleton instance
     *
     * @var isys_auth_document
     */
    private static $m_instance;

    /**
     * @var array
     */
    private $allowedDocuments;

    /**
     * @var array
     */
    private $allowedTemplates;

    /**
     * Method for returning the available auth-methods. This will be used for the GUI.
     *
     * @return array
     * @author Selcuk Kekec <skekec@i-doit.com>
     */
    public function get_auth_methods()
    {
        return [
            'documents'               => [
                'title'  => 'LC__AUTH_GUI__DOCUMENTS',
                'type'   => 'documents',
                'rights' => [
                    isys_auth::CREATE,
                    isys_auth::VIEW,
                    isys_auth::DELETE,
                    isys_auth::EDIT
                ]
            ],
            'documents_in_categories' => [
                'title'  => 'LC__AUTH_GUI__DOCUMENTS_IN_CATEGORIES',
                'type'   => 'documents_in_categories',
                'rights' => [
                    isys_auth::CREATE,
                    isys_auth::VIEW,
                    isys_auth::DELETE,
                    isys_auth::EDIT
                ]
            ],
            'templates'               => [
                'title'  => 'LC__AUTH_GUI__TEMPLATES',
                'type'   => 'document_templates',
                'rights' => [
                    isys_auth::CREATE,
                    isys_auth::VIEW,
                    isys_auth::DELETE,
                    isys_auth::EDIT
                ]
            ],
            'templates_in_categories' => [
                'title'  => 'LC__AUTH_GUI__TEMPLATES_IN_CATEGORIES',
                'type'   => 'templates_in_categories',
                'rights' => [
                    isys_auth::CREATE,
                    isys_auth::VIEW,
                    isys_auth::DELETE,
                    isys_auth::EDIT
                ]
            ],
            'online-repository'       => [
                'title'  => 'LC__MODULE__DOCUMENT__ONLINE_REPOSITORY',
                'type'   => 'boolean',
                'rights' => [
                    isys_auth::VIEW
                ]
            ]
        ];
    }

    /**
     * Method for retrieving the "parameter" in the configuration GUI. Gets called generically by "ajax()" method.
     *
     * @see    isys_module_auth->ajax_retrieve_parameter();
     *
     * @param  string  $authMethod
     * @param  string  $authParameter
     * @param  integer $counter
     * @param  boolean $editMode
     * @param  boolean $combination This parameter is used, when more than one box is displayed at once (category in object, ...).
     *
     * @return array
     * @author Leonard Fischer <lfischer@synetics.de>
     */
    public function retrieve_parameter($authMethod, $authParameter, $counter, $editMode = false, $combination = false)
    {
        $database = isys_application::instance()->container->get('database');
        $template = isys_application::instance()->container->get('template');
        $language = isys_application::instance()->container->get('language');

        $l_return = [
            'html'    => '',
            'method'  => $authMethod,
            'param'   => $authParameter,
            'counter' => $counter
        ];

        switch ($authMethod) {
            case 'documents':
                $dialogData = [];
                $dialogPlugin = new isys_smarty_plugin_f_dialog();
                $result = isys_document_dao::instance($database)->get_data();

                while ($l_row = $result->get_row()) {
                    $dialogData[$l_row['isys_document__id']] = $language->get($l_row['isys_document__title']) .
                        ' (' . $language->get($l_row['isys_document_template__title']) .
                        $language->get('LC__MODULE__DOCUMENT__FOR') .
                        $language->get($l_row['isys_obj__title']) . ')';
                }

                $parameters = [
                    'name'              => 'auth_param_form_' . $counter,
                    'p_arData'          => $dialogData,
                    'p_editMode'        => $editMode,
                    'p_bDbFieldNN'      => 1,
                    'p_bInfoIconSpacer' => 0,
                    'p_strClass'        => 'input-small',
                    'p_strSelectedID'   => $authParameter
                ];

                $l_return['html'] = $dialogPlugin->navigation_edit($template, $parameters);
                break;

            case 'documents_in_categories':
            case 'templates_in_categories':
                $dialogData = [];
                $dialogPlugin = new isys_smarty_plugin_f_dialog();
                $result = isys_factory_cmdb_dialog_dao::get_instance('isys_document_type', $database)->get_data();

                if (is_array($result) && count($result)) {
                    foreach ($result as $l_row) {
                        $dialogData[$l_row['isys_document_type__id']] = $language->get($l_row['isys_document_type__title']);
                    }
                }

                $parameters = [
                    'name'              => 'auth_param_form_' . $counter,
                    'p_arData'          => $dialogData,
                    'p_editMode'        => $editMode,
                    'p_bDbFieldNN'      => 1,
                    'p_bInfoIconSpacer' => 0,
                    'p_strClass'        => 'input-small',
                    'p_strSelectedID'   => $authParameter
                ];

                $l_return['html'] = $dialogPlugin->navigation_edit($template, $parameters);
                break;

            case 'document_templates':
                $dialogData = [];
                $dialogPlugin = new isys_smarty_plugin_f_dialog();
                $result = isys_document_dao_templates::instance($database)
                    ->get_data();

                while ($l_row = $result->get_row()) {
                    $dialogData[$l_row['isys_document_template__id']] = $language->get($l_row['isys_document_template__title']);
                }

                $parameters = [
                    'name'              => 'auth_param_form_' . $counter,
                    'p_arData'          => $dialogData,
                    'p_editMode'        => $editMode,
                    'p_bDbFieldNN'      => 1,
                    'p_bInfoIconSpacer' => 0,
                    'p_strClass'        => 'input-small',
                    'p_strSelectedID'   => $authParameter
                ];

                $l_return['html'] = $dialogPlugin->navigation_edit($template, $parameters);
                break;
        }

        return $l_return;
    }

    /**
     * Use generic logic.
     *
     * @param  integer $right
     * @param  integer $id
     *
     * @return boolean
     * @throws isys_exception_auth
     */
    public function documents_in_categories($right, $id = null)
    {
        if (!$this->is_auth_active()) {
            return true;
        }

        return $this->generic_right(
            $right,
            'documents_in_categories',
            $id,
            new isys_exception_auth(_L('LC__AUTH__MANUAL_EXCEPTION__MISSING_RIGHT_FOR_DOCUMENTS'))
        );
    }

    /**
     * Method for checking, if the user is allowed to open and execute document operations.
     *
     * @param  integer $right
     * @param  integer $id
     *
     * @return boolean
     * @throws isys_exception_auth
     */
    public function documents($right, $id = null)
    {
        if (!$this->is_auth_active()) {
            return true;
        }

        if ($id === null && $this->has('documents')) {
            return true;
        }

        if ($id !== null && isset($this->allowedDocuments[self::WILDCHAR]) && in_array($right, $this->allowedDocuments[self::WILDCHAR], false)) {
            return true;
        }

        if ($id !== null && isset($this->allowedDocuments[$id]) && in_array($right, $this->allowedDocuments[$id], false)) {
            return true;
        }

        return $this->generic_right(
            $right,
            'documents',
            $id,
            new isys_exception_auth(_L('LC__AUTH__MANUAL_EXCEPTION__MISSING_RIGHT_FOR_DOCUMENTS'))
        );
    }

    /**
     * Use generic logic.
     *
     * @param  integer $right
     * @param  integer $id
     *
     * @return boolean
     * @throws isys_exception_auth
     */
    public function templates_in_categories($right, $id = null)
    {
        if (!$this->is_auth_active()) {
            return true;
        }

        return $this->generic_right(
            $right,
            'templates_in_categories',
            $id,
            new isys_exception_auth(_L('LC__AUTH__MANUAL_EXCEPTION__MISSING_RIGHT_FOR_TEMPLATES'))
        );
    }

    /**
     * Method for checking, if the user is allowed to execut template operations.
     *
     * @param  integer $right
     * @param  integer $id
     *
     * @return boolean
     * @throws isys_exception_auth
     * @author Selcuk Kekec <skekec@i-doit.com>
     */
    public function templates($right, $id = null)
    {
        if (!$this->is_auth_active()) {
            return true;
        }

        if ($id === null && $this->has('templates')) {
            return true;
        }

        if ($id !== null && isset($this->allowedTemplates[self::WILDCHAR]) && in_array($right, $this->allowedTemplates[self::WILDCHAR], false)) {
            return true;
        }

        if ($id !== null && isset($this->allowedTemplates[$id]) && in_array($right, $this->allowedTemplates[$id], false)) {
            return true;
        }

        return $this->generic_right(
            $right,
            'templates',
            $id,
            new isys_exception_auth(_L('LC__AUTH__MANUAL_EXCEPTION__MISSING_RIGHT_FOR_TEMPLATES'))
        );
    }

    /**
     * Method for checking online repository rights.
     *
     * @param  integer $p_right
     *
     * @return boolean
     * @throws isys_exception_auth
     */
    public function repository($p_right)
    {
        if (!$this->is_auth_active()) {
            return true;
        }

        return $this->generic_right(
            $p_right,
            'online-repository',
            self::EMPTY_ID_PARAM,
            new isys_exception_auth(_L('LC__AUTH__MANUAL_EXCEPTION__MISSING_RIGHT_FOR_ONLINE_REPOSITORY'))
        );
    }

    /**
     * Get ID of related module.
     *
     * @return integer
     */
    public function get_module_id()
    {
        return C__MODULE__DOCUMENT;
    }

    /**
     * Get title of related module.
     *
     * @return string
     */
    public function get_module_title()
    {
        return 'LC__MODULE__DOCUMENT';
    }

    /**
     * Constructor, will load all necessary paths.
     */
    protected function __construct()
    {
        parent::__construct();

        $this->allowedDocuments = $this->m_paths['documents'] ?? [];
        $this->allowedTemplates = $this->m_paths['templates'] ?? [];

        // Get all allowed documents by categories.
        if ($this->has('documents_in_categories')) {
            // @see DOKU-526 Go sure to handle arrays.
            $documentsInCategories = $this->m_paths['documents_in_categories'] ?? [];

            if (in_array(self::WILDCHAR, array_keys($documentsInCategories), true)) {
                $this->allowedDocuments[self::WILDCHAR] = $documentsInCategories[self::WILDCHAR];
            }

            $sql = 'SELECT isys_document__id AS id, isys_document__isys_document_type__id AS categoryId
                FROM isys_document
                WHERE isys_document__isys_document_type__id ' . self::$m_dao->prepare_in_condition(array_keys($documentsInCategories)) . ';';

            $result = self::$m_dao->retrieve($sql);

            while ($row = $result->get_row()) {
                if (!isset($this->allowedDocuments[$row['id']])) {
                    $this->allowedDocuments[$row['id']] = [];
                }

                $this->allowedDocuments[$row['id']] = array_merge(
                    (array)$this->allowedDocuments[$row['id']],
                    (array)$documentsInCategories[$row['categoryId']]
                );
            }
        }

        // Get all allowed templates by categories.
        if ($this->has('templates_in_categories')) {
            // @see DOKU-526 Go sure to handle arrays.
            $templatesInCategories = $this->m_paths['templates_in_categories'] ?? [];

            if (in_array(self::WILDCHAR, array_keys($templatesInCategories), true)) {
                $this->allowedTemplates[self::WILDCHAR] = $templatesInCategories[self::WILDCHAR];
            }

            $sql = 'SELECT isys_document_template__id AS id, isys_document_template__isys_document_type__id AS categoryId
                FROM isys_document_template
                WHERE isys_document_template__isys_document_type__id ' . self::$m_dao->prepare_in_condition(array_keys($templatesInCategories)) . ';';

            $result = self::$m_dao->retrieve($sql);

            while ($row = $result->get_row()) {
                if (!isset($this->allowedTemplates[$row['id']])) {
                    $this->allowedTemplates[$row['id']] = [];
                }

                $this->allowedTemplates[$row['id']] = array_merge(
                    (array)$this->allowedTemplates[$row['id']],
                    (array)$templatesInCategories[$row['categoryId']]
                );
            }
        }
    }

    /**
     * Retrieve singleton instance of authorization class.
     *
     * @return isys_auth_document
     * @author Selcuk Kekec <skekec@i-doit.com>
     */
    public static function instance()
    {
        // If the DAO has not been loaded yet, we initialize it now.
        if (self::$m_dao === null) {
            self::$m_dao = new isys_auth_dao(isys_application::instance()->container->get('database'));
        }

        if (self::$m_instance === null) {
            self::$m_instance = new self;
        }

        return self::$m_instance;
    }
}
